﻿using log4net;
using Microsoft.AspNetCore.Http;
using Microsoft.AspNetCore.Mvc;
using System;
using System.Threading.Tasks;
using VA.PPMS.IWS.Api.Configuration.Interface;
using VA.PPMS.IWS.Api.Data;
using VA.PPMS.IWS.Api.Helpers;
using VA.PPMS.IWS.Api.HttpClientHandler.Interface;

namespace VA.PPMS.IWS.Api.Controllers
{
    [Route("OperationOutcome/v1")]
    public class DasOperationOutcomeController : Controller
    {
        private readonly ILog _logger;
        private readonly IIwsConfiguration _configuration;
        private readonly IHttpClientHandler _client;

        public DasOperationOutcomeController(ILog logger, IIwsConfiguration configuration, IHttpClientHandler client)
        {
            _logger = logger;
            _configuration = configuration;
            _client = client;
        }

        [HttpPost]
        public async Task<IActionResult> Post([FromBody]string fhirMessage)
        {
            string conversationId = null;

            try
            {
                _logger.Info("@@@@@ Info - Start POST for DasOperationOutcomeController @@@@@");

                var message = DasHelper.ParseHeaders(Request);
                var outcome = FhirHelper.ConvertToOutcome(fhirMessage);
                conversationId = message.ConversationId;

                if (outcome == null)
                {
                    _logger.Error("DasOperationOutcomeController: OperationOutcome payload is missing");
                    return StatusCode(StatusCodes.Status400BadRequest);
                }

                var hasOutcomeId = !string.IsNullOrEmpty(outcome.Id);
                if (hasOutcomeId && message.IsValid)
                {
                    _logger.Info($"-- DasOperationOutcome - start process data: {outcome.Id}");

                    var success = (outcome.Id == FhirHelper.AllOk);
                    message.Content = success.ToString();

                    using (var response = await _client.PostAsync(await _configuration.GetDasOperationOutcomeUri(), message))
                    {
                        if (response.IsSuccessStatusCode)
                        {
                            _logger.Info($"-- DasOperationOutcome - end process data: {outcome.Id}");

                            _logger.Info("@@@@@ Info - End POST for DasOperationOutcomeController @@@@@");
                            return StatusCode(StatusCodes.Status201Created, DasHelper.StatusCode201(conversationId));
                        }

                        _logger.Error($"DasOperationOutcomeController POST Exception. There was an error posting to the Function API: {response}");
                        var error = await response.Content.ReadAsStringAsync();
                        _logger.Error($"DasOperationOutcomeController POST Exception. The error is {error}: {response}");

                        return StatusCode(StatusCodes.Status400BadRequest, DasHelper.StatusCode400(conversationId, message.ReceiverId));
                    }
                }

                _logger.Error("CcnInitialOutcomeController: DasOperationOutcomeController is missing data required to process the information");
                return StatusCode(StatusCodes.Status400BadRequest, DasHelper.StatusCode400(conversationId, message.ReceiverId));
            }
            catch (Exception ex)
            {
                _logger.Error("DasOperationOutcomeController POST Exception:", ex);
                return StatusCode(StatusCodes.Status500InternalServerError, DasHelper.StatusCode500(conversationId));
            }
        }
    }
}